# Visualization

## Introduction

Pawpaw offers basic visualization tools intended to help develop, test, and debug.

## Format Strings

Pawpaw's ``Ito`` has a well-defined  ``.__format__`` that makes it possible to specify a format string for an ``Ito`` object in [formatted string literals](https://docs.python.org/3/reference/lexical_analysis.html#f-strings) and when using ``str.format()``.  Two categories of formatting directives are available: Integer and String, each with its own vocabulary and modifiers.  Both formatting directives can occur within a single f-string expression.

### Integer Expressions

An integer expression has the following form:

```
integer expression := directive[:[[fill]align][sign][#][0][width][grouping_option][type]]
```

The *directive* field has one of the following values:

| Directive | Meaning            | Example      |
| :---:     |--------------------|--------------|
| ``'%span'`` | ``.span`` as tuple | ``'(1, 14)'`` |
| ``'%start'`` | ``.start``         | ``'1'``      |
| ``'%stop'`` | ``.stop``          | ``'14'``   |

The *fill*, *align*, *sign*, *width*, *grouping_option*, and *type* fields, along with the characters ``'#'`` and ``'0'`` have identical form and behavior as Python's [Format Specification Mini-Language](https://docs.python.org/3/library/string.html?highlight=fill%20align%20sign%20width#format-specification-mini-language). Conversely, the *precision* field and ``'z'`` character[^z_field] are not supported because all ``Ito`` integer directives yield only non-negative integers.

In the case of ``'%span'``, fields are applied to both the values and the result is returned as a Python tuple ``'(x, y)'``.

You may surround integer expressions with additional text as needed, however, if the surrounding text contains characters that could be mis-interpreted as part of the integer expression, you must insert the zero-length whitespace directive, ``'% '`` to allow the parser to distinguish such text from the attached expression.

#### Examples

```python
>>> from pawpaw import Ito
>>> i = Ito(' Hello, world! ', 1, -1, 'phrase')
>>> f'{i:%span}'  # Just .span
f'{i:%span}'  # Just .span
'(1, 14)'
>>> f'{i:%span:~^4}'  # Add some formatting
'(~1~~, ~14~)'
>>> f'{i:[%span:~^4]}'  # Surround with brackets... ok
'[(~1~~, ~14~)]'
>>> f'{i:_%span:~^4_}'  # Now surround with underscores... right underscore will be interpreted as grouping_option
'_(~1~~, ~14~)'
>>> f'{i:_%span:~^4% _}'  # Fix by adding zero-length whitespace directive
'_(~1~~, ~14~)_'
```

### String Expressions

While Pawpaw's Ito integer expressions are nearly identical to Python's, string expressions have many key differences.  A Pawpaw
string expresssion has the form:

```
string expression := directive[![lslice]conversion[rslice]][:[abbr_pos]width[abbr]]
```

### **directive**

The *directive* field indicates which ``str`` value you wish to format, and has one of the following values:

|   Directive   |                             Meaning                             |         Example         |
|:-------------:|:---------------------------------------------------------------:|:-----------------------:|
| ``'%string'`` |                           ``.string``                           |  ``' Hello, world! '``  |
|  ``'%desc'``  |                            ``.desc``                            |      ``'phrase'``       |
| ``'%substr'`` |                         ``.__str__()``                          |   ``'Hello, world!'``   |
| ``'%value'``  |                        ``str(.value())``                        |   ``'Hello, world!'``   |
|   ``'% '``    |  zero-length whitespace;</br>used to define expression parsing  |         ``''``          |

#### Examples

```python
>>> from pawpaw import Ito
>>> i = Ito('ab12cd', 2, -2, 'phrase')
>>> f'{i:%desc}'  # .desc
phrase
>>> f'{i:%string}'
'ab12cd'
>>> f'{i:%substr}'
'12'
```

The zero-length whitespace directive is used as a null-separator to clarify ambiguous format expressions.

### *lslice*, *rslice*, & *conversion*

The optional *conversion* field indicates what method, if any, to pass the *directive* result to.  It has two possible values:

| Value | Meaning     | Example |
| :---: | :---------: | :--     |
| ``'a'`` | ``ascii`` | ``'\n'`` → ``"'\\n'"`` |
| ``'r'`` | ``repr``  | ``'\n'`` → ``'\n'`` |

Note that unlike Python's ``str`` format syntax, there is no ``s`` conversion value.  This is because all *directive* values
already return ``str`` values.

The *lslice* and *rslice* are optional integer values that, when present, slice the left and right of the conversion result.
The reason for these fields is based on the behavior of ``ascii()`` and ``repr()`` on a ``str``:

```python
>>> s = 'a\nb'  
>>> print(s)  # embedded newline shows in results
a
b
>>> print(ascii(s))  # embedded newline replaced with '\n', however, apostrophes are prepended & appended
'a\nb'i
>>> print(repr(s))  # similar behavior to ascii()
'a\nb'
```

Calling ``repr()`` is particularly useful if you need to normalize a str for rendering.  However, the intent of this method is
not strictly for visualization.  From the Python docs, ``repr()``:

> this function makes an attempt to return a string that would yield an object with the same value when passed to eval()

This allows code such as the following:

```python
>>> s2 = eval(repr(s))
>>> s == s2
True
```

Pawpaw's *lslice* and *rslice* allows you to remove a given number of characters from either the left and/or right of a
conversion result.  For example, to remove the added apostrophes resulting from a call to ``repr()``:

```python
>>> from pawpaw import Ito
>>> i = Ito('a\nb\c')
>>> print(f'{i%substr!1r1}')
a\nb\c
```

When included, *lslice* and *rslice* will typically have a value of 1.  However, other possible values are allowed in order to support
arbitrary slicing of the result of the ``'%value'`` *directive*, whose result may include additional characters that need removal.

### *abbr_pos*, *width*, & *abbr*

Unlike Python's format string language, the *width* field indicates the **_maximum_ width** for the output value, which is shortened
as needed.  This is particularly useful for ``Ito`` trees, whose parent nodes may have long substrings.  Specifying a maximum width
allows output to be shortened to aid visualization.

The *abbr_pos* and *abbr* fields, if present, allow an abbreviation string to be incorporated into the result, to visually
indicate that the output was shortened.  If *width* is longer than the length of the *abbr*, then *abbr* itself gets truncated to
the value of *width*, and is returned as the result  Note that you can't use a digit as a value for *abbr*.  If you attempt this,
it will be interpreted as part of the decimal value for *width* instead.

The *abbr_pos* field indicates both a) how the string is shortened and b) where *abbr* appears in the result The allowable values
are:

| Value | Shorten Method | Location of *abbr* | Example |
| :---: | :------------: | :----------------: | ------- |
| ``'<'`` | right trim | prefix | ``'abcdef'`` → ``'…cdef'`` |
| ``'^'`` | delete from middle | middle | ``'abcdef'`` → ``'ab…ef'`` |
| ``'>'`` | left trim | suffix *(default)* | ``'abcdef'`` → ``'abcd…'`` |

As with integer expressions, you may surround your expressions with additional text as needed, however, you will need to use the
zero-length whitespace directive, ``'% '`` to allow the parser to distinguish when such text is not part of the conjoined expression.

```python
>>> from pawpaw import Ito
>>> i = Ito(' Hello, world! ', 1, -1, 'phrase')
>>> f'{i:%substr:6…(i)}'  # entire phrase '…(i)' interpreted as abbr_suffix
'He…(i)'
>>> f'{i:%substr:6…(i)}'  # use zero-length whitespace directive to define end of prior directive
'Hello…(i)'
```

### Min-width behavior

Note that if you need the normal Python min-width behavior, simply use the str value (rather than an ito) in your format
string:

```python
>>> from pawpaw import Ito
>>> i = Ito(' Hello, world! ', 1, -1, 'phrase')
>>> f'{str(i):~^17}'  # Python format str expression: min-width
'~~Hello, world!~~'
>>> f'{i:%substr:4…}'  # Pawpaw format Ito expression: max-width
'Hel…'
```

## ``pepo``

The ``pepo``[^pepo] module featrues a variety of visualization outputs for quickly inspecting pawpaw trees.

### ``Compact``

The class ``Compact`` walks a pawpaw tree and generates a string whose lines correspond the constituent itos.  Lines are numbered and indented to visually inspect the tree:

```python
>>> import pawpaw
>>> import regex
>>> root = pawpaw.Ito('See Jack run.', desc='phrase')
>>> re = regex.compile(r'(?<word>(?<char>[a-z])+)', regex.IGNORECASE)
>>> root.children.add(*pawpaw.Ito.from_re(re, root))
>>> v_compact = pawpaw.visualization.pepo.Compact()
>>> print(v_compact.dumps(root))
1: (0, 13) 'phrase' : 'See Jack run.'
    1: (0, 3) '0' : 'See'
        1: (0, 3) 'word' : 'See'
            1: (0, 1) 'char' : 'S'
            2: (1, 2) 'char' : 'e'
            3: (2, 3) 'char' : 'e'
    2: (4, 8) '0' : 'Jack'
        1: (4, 8) 'word' : 'Jack'
            1: (4, 5) 'char' : 'J'
            2: (5, 6) 'char' : 'a'
            3: (6, 7) 'char' : 'c'
            4: (7, 8) 'char' : 'k'
    3: (9, 12) '0' : 'run'
        1: (9, 12) 'word' : 'run'
            1: (9, 10) 'char' : 'r'
            2: (10, 11) 'char' : 'u'
            3: (11, 12) 'char' : 'n'
```

### ``Tree``

The class ``Tree`` walks a pawpaw tree and generates a a string whose lines correspond the constituent itos.  Lines are prefixed with lines that visually portray ancestry, similar to the tree command[^tree_command].

```python
>>> import pawpaw
>>> import regex
>>> root = pawpaw.Ito('See Jack run.', desc='phrase')
>>> re = regex.compile(r'(?<word>(?<char>[a-z])+)', regex.IGNORECASE)
>>> root.children.add(*pawpaw.Ito.from_re(re, root))
>>> v_tree = pawpaw.visualization.pepo.Tree()
>>> print(v_tree.dumps(root))
(0, 13) 'phrase' : 'See Jack run.'
├──(0, 3) '0' : 'See'
│  └──(0, 3) 'word' : 'See'
│     ├──(0, 1) 'char' : 'S'
│     ├──(1, 2) 'char' : 'e'
│     └──(2, 3) 'char' : 'e'
├──(4, 8) '0' : 'Jack'
│  └──(4, 8) 'word' : 'Jack'
│     ├──(4, 5) 'char' : 'J'
│     ├──(5, 6) 'char' : 'a'
│     ├──(6, 7) 'char' : 'c'
│     └──(7, 8) 'char' : 'k'
└──(9, 12) '0' : 'run'
   └──(9, 12) 'word' : 'run'
      ├──(9, 10) 'char' : 'r'
      ├──(10, 11) 'char' : 'u'
      └──(11, 12) 'char' : 'n'
```

### ``Json``

The class ``Json`` generates JSON suitable for visual inspection of a pawpaw tree.  Note that the pawpaw.serialziation module provides two json encoders whose JSON is more suitable for serialization.  See [Serialization](./8.%20Serialization.md) in the docs for more details.

```python
>>> import pawpaw
>>> import regex
>>> root = pawpaw.Ito('See Jack run.', desc='phrase')
>>> re = regex.compile(r'(?<word>(?<char>[a-z])+)', regex.IGNORECASE)
>>> root.children.add(*pawpaw.Ito.from_re(re, root))
>>> v_json = pawpaw.visualization.pepo.Json()
>>> print(v_json.dumps(root))
{
    "itos": [
        {
            "start": 0,
            "stop": 13,
            "desc": "phrase",
            "substr": "See Jack run.",
            "children": [
                {
                    "start": 0,
                    "stop": 3,
                    "desc": "0",
                    "substr": "See",
                    "children": [
                        {
                            "start": 0,
                            "stop": 3,
                            "desc": "word",
                            "substr": "See",
                            "children": [
                                {
                                    "start": 0,
                                    "stop": 1,
                                    "desc": "char",
                                    "substr": "S",
                                    "children": []
                                },
                                {
                                    "start": 1,
                                    "stop": 2,
                                    "desc": "char",
                                    "substr": "e",
                                    "children": []
                                },
                                {
                                    "start": 2,
                                    "stop": 3,
                                    "desc": "char",
                                    "substr": "e",
                                    "children": []
                                }
                            ]
                        }
                    ]
                },
                {
                    "start": 4,
                    "stop": 8,
                    "desc": "0",
                    "substr": "Jack",
                    "children": [
                        {
                            "start": 4,
                            "stop": 8,
                            "desc": "word",
                            "substr": "Jack",
                            "children": [
                                {
                                    "start": 4,
                                    "stop": 5,
                                    "desc": "char",
                                    "substr": "J",
                                    "children": []
                                },
                                {
                                    "start": 5,
                                    "stop": 6,
                                    "desc": "char",
                                    "substr": "a",
                                    "children": []
                                },
                                {
                                    "start": 6,
                                    "stop": 7,
                                    "desc": "char",
                                    "substr": "c",
                                    "children": []
                                },
                                {
                                    "start": 7,
                                    "stop": 8,
                                    "desc": "char",
                                    "substr": "k",
                                    "children": []
                                }
                            ]
                        }
                    ]
                },
                {
                    "start": 9,
                    "stop": 12,
                    "desc": "0",
                    "substr": "run",
                    "children": [
                        {
                            "start": 9,
                            "stop": 12,
                            "desc": "word",
                            "substr": "run",
                            "children": [
                                {
                                    "start": 9,
                                    "stop": 10,
                                    "desc": "char",
                                    "substr": "r",
                                    "children": []
                                },
                                {
                                    "start": 10,
                                    "stop": 11,
                                    "desc": "char",
                                    "substr": "u",
                                    "children": []
                                },
                                {
                                    "start": 11,
                                    "stop": 12,
                                    "desc": "char",
                                    "substr": "n",
                                    "children": []
                                }
                            ]
                        }
                    ]
                }
            ]
        }
    ]
}
```

### ``Xml``

The class ``Xml`` generates XML suitable for visual inspection of a pawpaw tree.

```python
>>> import pawpaw
>>> import regex
>>> root = pawpaw.Ito('See Jack run.', desc='phrase')
>>> re = regex.compile(r'(?<word>(?<char>[a-z])+)', regex.IGNORECASE)
>>> root.children.add(*pawpaw.Ito.from_re(re, root))
>>> v_xml = pawpaw.visualization.pepo.Xml()
>>> print(v_xml.dumps(root))
<?xml version="1.0" encoding="UTF-8" ?>
<itos>
    <ito start="0" stop="13" desc="phrase">
    <substring>See Jack run.</substring>
    <children>
        <ito start="0" stop="3" desc="0">
        <substring>See</substring>
        <children>
            <ito start="0" stop="3" desc="word">
            <substring>See</substring>
            <children>
                <ito start="0" stop="1" desc="char">
                <substring>S</substring>
            </ito>
                <ito start="1" stop="2" desc="char">
                <substring>e</substring>
            </ito>
                <ito start="2" stop="3" desc="char">
                <substring>e</substring>
            </ito>
            </children>
        </ito>
        </children>
    </ito>
        <ito start="4" stop="8" desc="0">
        <substring>Jack</substring>
        <children>
            <ito start="4" stop="8" desc="word">
            <substring>Jack</substring>
            <children>
                <ito start="4" stop="5" desc="char">
                <substring>J</substring>
            </ito>
                <ito start="5" stop="6" desc="char">
                <substring>a</substring>
            </ito>
                <ito start="6" stop="7" desc="char">
                <substring>c</substring>
            </ito>
                <ito start="7" stop="8" desc="char">
                <substring>k</substring>
            </ito>
            </children>
        </ito>
        </children>
    </ito>
        <ito start="9" stop="12" desc="0">
        <substring>run</substring>
        <children>
            <ito start="9" stop="12" desc="word">
            <substring>run</substring>
            <children>
                <ito start="9" stop="10" desc="char">
                <substring>r</substring>
            </ito>
                <ito start="10" stop="11" desc="char">
                <substring>u</substring>
            </ito>
                <ito start="11" stop="12" desc="char">
                <substring>n</substring>
            </ito>
            </children>
        </ito>
        </children>
    </ito>
    </children>
</ito>
<itos>
```

## ``highlighter``

The ``highlighter`` module contains a single class, ``Highlighter``.  This class offers quick visualization a tree by displaying colorizing the background color of hte underlying text for segments present in an Ito.

```python
>>> import pawpaw
>>> import regex
>>> root = Ito('See Jack run.', desc='phrase')
>>> re = regex.compile(r'(?<word>(?<char>[a-z])+)', regex.IGNORECASE)
>>> root.children.add(*pawpaw.Ito.from_re(re, root))
>>> palette = pawpaw.visualization.sgr.palettes.PAWPAW
>>> v_highlighter = pawpaw.visualization.Highlighter(palette)
>>> v_highlighter.print(root)
```

The background color is rotated through the palette at each Ito boundary.  Note that it is not possible, for an arbitrary palette and tree, to guarrantee that the span for a given parent Ito will be the same.  For example, given a two-color palette and the following tree:

```text
A-------------A     # Assign color 1
    B------B        # Assign color 2; boundary AB and BA are visible
    C---C           # If assign color 1, boundary AC is invisible; if assign color 2, boundary CB is invisible
```

Therefore, when using ``Highlighter``, keep in mind that the color shifts only indicate Ito boundaries, and are no indicative of type or identity.

## ``ascii_box``

The ``ascii_box`` module support ascii box drawing charcter output...

[^z_field]: The ``'z'`` field was added in Python 3.11 and coerces negative zero floating-point values to positive zero after rounding to the format precision  See [PEP 682](https://peps.python.org/pep-0682/) for details.

[^pepo]: **pepo**: *A fleshy, several-seeded fruit that has developed from one flower having a single ovary divided into several carpels, which develops a firm or tough rind as it matures*; for example, a melon, pumpkin, mango, banana... or pawpaw!

[^tree_command]: The ``tree`` is a recursive directory listing command command, present in Unix/Linux, MS-DOS, and Widnows.  See [tree (command)](https://en.wikipedia.org/wiki/Tree_(command)) for more info.
